<?php
require_once __DIR__ . '/../includes/functions.php';

$currentUser = getCurrentUser();

// 如果没有登录，跳转到登录页
if (!$currentUser) {
    header('Location: ' . SITE_URL . '/mobile/login.php');
    exit;
}

$db = Database::getInstance();
$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'profile') {
        // 修改个人资料
        $email = trim($_POST['email'] ?? '');
        $signature = trim($_POST['signature'] ?? '');
        
        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error = '请输入有效的邮箱地址';
        } elseif (strlen($signature) > 500) {
            $error = '个性签名不能超过500个字符';
        } else {
            // 检查邮箱是否被其他用户使用
            $existing = $db->queryOne(
                "SELECT id FROM users WHERE email = ? AND id != ?",
                [$email, $currentUser['id']]
            );
            
            if ($existing) {
                $error = '该邮箱已被其他用户使用';
            } else {
                $db->execute(
                    "UPDATE users SET email = ?, signature = ? WHERE id = ?",
                    [$email, $signature, $currentUser['id']]
                );
                $success = '个人资料已更新';
                // 刷新用户信息
                $currentUser = getCurrentUser();
            }
        }
    } elseif ($action === 'password') {
        // 修改密码
        $oldPassword = $_POST['old_password'] ?? '';
        $newPassword = $_POST['new_password'] ?? '';
        $confirmPassword = $_POST['confirm_password'] ?? '';
        
        if (empty($oldPassword) || empty($newPassword)) {
            $error = '请填写所有密码字段';
        } elseif (strlen($newPassword) < 6) {
            $error = '新密码长度至少为6位';
        } elseif ($newPassword !== $confirmPassword) {
            $error = '两次输入的新密码不一致';
        } elseif (!password_verify($oldPassword, $currentUser['password'])) {
            $error = '当前密码错误';
        } else {
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            $db->execute(
                "UPDATE users SET password = ? WHERE id = ?",
                [$hashedPassword, $currentUser['id']]
            );
            $success = '密码已修改';
        }
    } elseif ($action === 'avatar') {
        // 上传头像（裁剪后的）
        $avatarData = $_POST['avatar_data'] ?? '';
        
        if (empty($avatarData)) {
            $error = '请先选择并裁剪图片';
        } else {
            // 解码 base64 图片数据
            $avatarData = str_replace('data:image/png;base64,', '', $avatarData);
            $avatarData = str_replace('data:image/jpeg;base64,', '', $avatarData);
            $avatarData = str_replace(' ', '+', $avatarData);
            $data = base64_decode($avatarData);
            
            if ($data === false) {
                $error = '图片数据解析失败';
            } else {
                // 生成文件名
                $filename = 'avatar_' . $currentUser['id'] . '_' . time() . '.png';
                $uploadPath = dirname(__DIR__) . '/assets/uploads/avatars/' . $filename;
                $webPath = 'assets/uploads/avatars/' . $filename;
                
                // 确保目录存在
                if (!is_dir(dirname($uploadPath))) {
                    mkdir(dirname($uploadPath), 0755, true);
                }
                
                // 删除旧头像
                if ($currentUser['avatar'] && file_exists(dirname(__DIR__) . '/' . $currentUser['avatar'])) {
                    @unlink(dirname(__DIR__) . '/' . $currentUser['avatar']);
                }
                
                // 保存图片
                if (file_put_contents($uploadPath, $data)) {
                    // 更新数据库
                    $db->execute(
                        "UPDATE users SET avatar = ? WHERE id = ?",
                        [$webPath, $currentUser['id']]
                    );
                    $success = '头像上传成功';
                    // 刷新用户信息
                    $currentUser = getCurrentUser();
                } else {
                    $error = '保存图片失败，请检查目录权限';
                }
            }
        }
    }
}

$pageTitle = '个人设置';
include __DIR__ . '/includes/header.php';
?>

<!-- 返回按钮 -->
<div class="mobile-back-bar">
    <a href="javascript:history.back();" class="mobile-back-btn">
        <i class="fas fa-arrow-left"></i> 返回
    </a>
</div>

<?php if ($error): ?>
<div class="mobile-alert mobile-alert-error">
    <i class="fas fa-exclamation-circle"></i> <?php echo e($error); ?>
</div>
<?php endif; ?>

<?php if ($success): ?>
<div class="mobile-alert mobile-alert-success">
    <i class="fas fa-check-circle"></i> <?php echo e($success); ?>
</div>
<?php endif; ?>

<!-- 头像设置 -->
<div class="mobile-section">
    <div class="mobile-section-title">
        <i class="fas fa-camera" style="color: var(--primary-color);"></i>
        <span>头像设置</span>
    </div>
    <div class="mobile-settings-avatar">
        <div class="mobile-current-avatar">
            <?php if (!empty($currentUser['avatar']) && $currentUser['avatar'] != 'NULL'): ?>
                <?php
                $avatarUrl = $currentUser['avatar'];
                if (!preg_match('/^https?:\/\//', $avatarUrl) && !preg_match('/^\//', $avatarUrl)) {
                    $avatarUrl = SITE_URL . '/' . $avatarUrl;
                }
                ?>
                <img src="<?php echo e($avatarUrl); ?>?t=<?php echo time(); ?>" alt="头像" id="current-avatar">
            <?php else: ?>
                <i class="fas fa-user" id="current-avatar-icon"></i>
            <?php endif; ?>
        </div>
        
        <div class="mobile-avatar-actions">
            <input type="file" id="avatar-input" accept="image/jpeg,image/png,image/gif,image/webp" style="display: none;">
            <button type="button" class="mobile-btn mobile-btn-primary" onclick="document.getElementById('avatar-input').click()">
                <i class="fas fa-folder-open"></i> 选择图片
            </button>
            <span class="mobile-avatar-hint">支持 JPG、PNG、GIF、WEBP 格式，最大 5MB</span>
        </div>
        
        <!-- 裁剪容器 -->
        <div id="cropper-container" style="display: none; margin-top: 15px;">
            <div style="max-width: 300px; margin: 0 auto;">
                <img id="cropper-image" src="" alt="待裁剪图片" style="max-width: 100%; display: block;">
            </div>
            <div style="display: flex; gap: 10px; justify-content: center; margin-top: 15px;">
                <button type="button" class="mobile-btn mobile-btn-primary" id="crop-btn">
                    <i class="fas fa-crop"></i> 确认裁剪
                </button>
                <button type="button" class="mobile-btn mobile-btn-outline" onclick="resetCropper()">
                    <i class="fas fa-times"></i> 取消
                </button>
            </div>
        </div>
        
        <!-- 裁剪预览 -->
        <div id="preview-container" style="display: none; margin-top: 15px; text-align: center;">
            <p style="color: #666; font-size: 13px; margin-bottom: 10px;">预览效果：</p>
            <div style="width: 100px; height: 100px; border-radius: 50%; overflow: hidden; margin: 0 auto; border: 2px solid var(--primary-color);">
                <img id="preview-image" src="" alt="预览" style="width: 100%; height: 100%; object-fit: cover;">
            </div>
        </div>
    </div>
    
    <!-- 隐藏表单用于提交裁剪后的图片 -->
    <form method="POST" action="" id="avatar-form" style="display: none;">
        <input type="hidden" name="action" value="avatar">
        <input type="hidden" name="avatar_data" id="avatar-data">
    </form>
</div>

<!-- 个人资料 -->
<div class="mobile-section">
    <div class="mobile-section-title">
        <i class="fas fa-user-edit" style="color: var(--primary-color);"></i>
        <span>个人资料</span>
    </div>
    <form method="POST" action="" class="mobile-form">
        <input type="hidden" name="action" value="profile">
        
        <div class="mobile-form-group">
            <label class="mobile-form-label">用户名</label>
            <input type="text" class="mobile-form-control" value="<?php echo e($currentUser['username']); ?>" disabled>
            <span class="mobile-form-hint">用户名不能修改</span>
        </div>
        
        <div class="mobile-form-group">
            <label class="mobile-form-label" for="email">邮箱</label>
            <input type="email" id="email" name="email" class="mobile-form-control" required
                   value="<?php echo e($currentUser['email'] ?? ''); ?>">
        </div>
        
        <div class="mobile-form-group">
            <label class="mobile-form-label" for="signature">个性签名</label>
            <textarea id="signature" name="signature" class="mobile-form-control" rows="3"
                      placeholder="写点什么来展示你自己..."><?php echo e($currentUser['signature'] ?? ''); ?></textarea>
            <span class="mobile-form-hint">最多500个字符</span>
        </div>
        
        <button type="submit" class="mobile-btn mobile-btn-primary mobile-btn-block">
            <i class="fas fa-save"></i> 保存资料
        </button>
    </form>
</div>

<!-- 修改密码 -->
<div class="mobile-section">
    <div class="mobile-section-title">
        <i class="fas fa-key" style="color: var(--primary-color);"></i>
        <span>修改密码</span>
    </div>
    <form method="POST" action="" class="mobile-form">
        <input type="hidden" name="action" value="password">
        
        <div class="mobile-form-group">
            <label class="mobile-form-label" for="old_password">当前密码</label>
            <input type="password" id="old_password" name="old_password" class="mobile-form-control" required>
        </div>
        
        <div class="mobile-form-group">
            <label class="mobile-form-label" for="new_password">新密码</label>
            <input type="password" id="new_password" name="new_password" class="mobile-form-control" required>
            <span class="mobile-form-hint">至少6位</span>
        </div>
        
        <div class="mobile-form-group">
            <label class="mobile-form-label" for="confirm_password">确认新密码</label>
            <input type="password" id="confirm_password" name="confirm_password" class="mobile-form-control" required>
        </div>
        
        <button type="submit" class="mobile-btn mobile-btn-primary mobile-btn-block">
            <i class="fas fa-key"></i> 修改密码
        </button>
    </form>
</div>

<!-- 引入 Cropper.js -->
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/cropperjs/1.5.13/cropper.min.css">
<script src="https://cdnjs.cloudflare.com/ajax/libs/cropperjs/1.5.13/cropper.min.js"></script>

<script>
let cropper = null;
const MAX_FILE_SIZE = 5 * 1024 * 1024; // 5MB

// 文件选择处理
document.getElementById('avatar-input').addEventListener('change', function(e) {
    const file = e.target.files[0];
    
    if (!file) return;
    
    // 检查文件大小
    if (file.size > MAX_FILE_SIZE) {
        alert('上传失败，图片大小超出上限（最大值：5MB）');
        this.value = '';
        return;
    }
    
    // 检查文件类型
    const allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    if (!allowedTypes.includes(file.type)) {
        alert('只支持 JPG、PNG、GIF、WEBP 格式的图片');
        this.value = '';
        return;
    }
    
    // 读取文件并显示裁剪界面
    const reader = new FileReader();
    reader.onload = function(e) {
        const image = document.getElementById('cropper-image');
        image.src = e.target.result;
        
        document.getElementById('cropper-container').style.display = 'block';
        document.getElementById('preview-container').style.display = 'none';
        
        // 初始化 Cropper
        if (cropper) {
            cropper.destroy();
        }
        
        cropper = new Cropper(image, {
            aspectRatio: 1,
            viewMode: 1,
            dragMode: 'move',
            autoCropArea: 0.8,
            restore: false,
            guides: true,
            center: true,
            highlight: false,
            cropBoxMovable: true,
            cropBoxResizable: true,
            toggleDragModeOnDblclick: false,
            ready: function() {
                updatePreview();
            },
            crop: function() {
                updatePreview();
            }
        });
    };
    reader.readAsDataURL(file);
});

// 更新预览
function updatePreview() {
    if (!cropper) return;
    
    const canvas = cropper.getCroppedCanvas({
        width: 300,
        height: 300,
        fillColor: '#fff',
        imageSmoothingEnabled: true,
        imageSmoothingQuality: 'high',
    });
    
    const previewImage = document.getElementById('preview-image');
    previewImage.src = canvas.toDataURL('image/png');
    document.getElementById('preview-container').style.display = 'block';
}

// 确认裁剪并上传
document.getElementById('crop-btn').addEventListener('click', function() {
    if (!cropper) return;
    
    const canvas = cropper.getCroppedCanvas({
        width: 300,
        height: 300,
        fillColor: '#fff',
        imageSmoothingEnabled: true,
        imageSmoothingQuality: 'high',
    });
    
    // 转换为 base64
    const imageData = canvas.toDataURL('image/png');
    
    // 检查裁剪后的大小
    const base64Length = imageData.length - 'data:image/png;base64,'.length;
    const sizeInBytes = (base64Length * 3) / 4;
    
    if (sizeInBytes > MAX_FILE_SIZE) {
        alert('裁剪后的图片仍然超出上限（最大值：5MB），请尝试压缩原图或选择更小的区域');
        return;
    }
    
    // 填充表单并提交
    document.getElementById('avatar-data').value = imageData;
    document.getElementById('avatar-form').submit();
});

// 重置裁剪器
function resetCropper() {
    if (cropper) {
        cropper.destroy();
        cropper = null;
    }
    document.getElementById('avatar-input').value = '';
    document.getElementById('cropper-container').style.display = 'none';
    document.getElementById('preview-container').style.display = 'none';
}
</script>

<style>
.mobile-back-bar {
    background: #fff;
    padding: 10px 15px;
    border-bottom: 1px solid #f0f0f0;
}
.mobile-back-btn {
    display: inline-flex;
    align-items: center;
    gap: 5px;
    color: #666;
    font-size: 14px;
    text-decoration: none;
}
.mobile-back-btn i {
    font-size: 12px;
}
.mobile-alert {
    padding: 12px 15px;
    margin: 10px;
    border-radius: 8px;
    font-size: 14px;
    display: flex;
    align-items: center;
    gap: 8px;
}
.mobile-alert-error {
    background: #ffebee;
    color: #c62828;
}
.mobile-alert-success {
    background: #e8f5e9;
    color: #2e7d32;
}
.mobile-settings-avatar {
    padding: 20px;
    text-align: center;
}
.mobile-current-avatar {
    width: 100px;
    height: 100px;
    margin: 0 auto 15px;
    border-radius: 50%;
    overflow: hidden;
    border: 3px solid #f0f0f0;
    display: flex;
    align-items: center;
    justify-content: center;
    background: #f5f5f5;
}
.mobile-current-avatar img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}
.mobile-current-avatar i {
    font-size: 50px;
    color: #ccc;
}
.mobile-avatar-actions {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 10px;
}
.mobile-avatar-hint {
    font-size: 12px;
    color: #999;
}
.mobile-form {
    padding: 15px;
}
.mobile-form-group {
    margin-bottom: 15px;
}
.mobile-form-label {
    display: block;
    font-size: 14px;
    color: #333;
    margin-bottom: 6px;
    font-weight: 500;
}
.mobile-form-control {
    width: 100%;
    padding: 12px;
    border: 1px solid #e0e0e0;
    border-radius: 8px;
    font-size: 14px;
    font-family: inherit;
    box-sizing: border-box;
}
.mobile-form-control:focus {
    outline: none;
    border-color: var(--primary-color);
}
.mobile-form-control:disabled {
    background: #f5f5f5;
    color: #999;
}
.mobile-form-hint {
    font-size: 12px;
    color: #999;
    margin-top: 4px;
    display: block;
}
.mobile-btn {
    padding: 12px 20px;
    border-radius: 8px;
    font-size: 14px;
    border: none;
    cursor: pointer;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 6px;
}
.mobile-btn-primary {
    background: var(--primary-color);
    color: white;
}
.mobile-btn-outline {
    background: #f5f5f5;
    color: #666;
    border: 1px solid #e0e0e0;
}
.mobile-btn-block {
    width: 100%;
}
</style>

<?php include __DIR__ . '/includes/footer.php'; ?>
