<?php
/**
 * 个人设置页面
 */
require_once __DIR__ . '/includes/functions.php';

// 检查是否登录
if (!isLoggedIn()) {
    setFlashMessage('error', '请先登录');
    redirect(SITE_URL . '/login.php');
}

$user = getCurrentUser();
$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'profile') {
        // 修改个人资料
        $email = trim($_POST['email'] ?? '');
        $signature = trim($_POST['signature'] ?? '');
        $csrfToken = $_POST['csrf_token'] ?? '';
        
        if (!validateCsrfToken($csrfToken)) {
            $error = '安全验证失败';
        } elseif (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error = '请输入有效的邮箱地址';
        } elseif (strlen($signature) > 500) {
            $error = '个性签名不能超过500个字符';
        } else {
            // 检查邮箱是否被其他用户使用
            $existing = db()->queryOne(
                "SELECT id FROM users WHERE email = ? AND id != ?",
                [$email, $user['id']]
            );
            
            if ($existing) {
                $error = '该邮箱已被其他用户使用';
            } else {
                db()->execute(
                    "UPDATE users SET email = ?, signature = ? WHERE id = ?",
                    [$email, $signature, $user['id']]
                );
                $success = '个人资料已更新';
            }
        }
    } elseif ($action === 'password') {
        // 修改密码
        $oldPassword = $_POST['old_password'] ?? '';
        $newPassword = $_POST['new_password'] ?? '';
        $confirmPassword = $_POST['confirm_password'] ?? '';
        $csrfToken = $_POST['csrf_token'] ?? '';
        
        if (!validateCsrfToken($csrfToken)) {
            $error = '安全验证失败';
        } elseif (empty($oldPassword) || empty($newPassword)) {
            $error = '请填写所有密码字段';
        } elseif (strlen($newPassword) < 6) {
            $error = '新密码长度至少为6位';
        } elseif ($newPassword !== $confirmPassword) {
            $error = '两次输入的新密码不一致';
        } elseif (!password_verify($oldPassword, $user['password'])) {
            $error = '当前密码错误';
        } else {
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            db()->execute(
                "UPDATE users SET password = ? WHERE id = ?",
                [$hashedPassword, $user['id']]
            );
            $success = '密码已修改，请重新登录';
        }
    } elseif ($action === 'avatar') {
        // 上传头像（裁剪后的）
        $csrfToken = $_POST['csrf_token'] ?? '';
        $avatarData = $_POST['avatar_data'] ?? '';
        
        if (!validateCsrfToken($csrfToken)) {
            $error = '安全验证失败';
        } elseif (empty($avatarData)) {
            $error = '请先选择并裁剪图片';
        } else {
            // 解码 base64 图片数据
            $avatarData = str_replace('data:image/png;base64,', '', $avatarData);
            $avatarData = str_replace('data:image/jpeg;base64,', '', $avatarData);
            $avatarData = str_replace(' ', '+', $avatarData);
            $data = base64_decode($avatarData);
            
            if ($data === false) {
                $error = '图片数据解析失败';
            } else {
                // 生成文件名
                $filename = 'avatar_' . $user['id'] . '_' . time() . '.png';
                $uploadPath = __DIR__ . '/assets/uploads/avatars/' . $filename;
                $webPath = 'assets/uploads/avatars/' . $filename;
                
                // 删除旧头像
                if ($user['avatar'] && file_exists(__DIR__ . '/' . $user['avatar'])) {
                    @unlink(__DIR__ . '/' . $user['avatar']);
                }
                
                // 保存图片
                if (file_put_contents($uploadPath, $data)) {
                    // 更新数据库
                    db()->execute(
                        "UPDATE users SET avatar = ? WHERE id = ?",
                        [$webPath, $user['id']]
                    );
                    $success = '头像上传成功';
                } else {
                    $error = '保存图片失败，请检查目录权限';
                }
            }
        }
    }
}

// 重新获取用户信息（可能已经更新）
$user = getCurrentUser();

$pageTitle = '个人设置';
include __DIR__ . '/templates/header.php';
?>

<div class="card">
    <div class="card-header">
        <h2><i class="fas fa-cog"></i> 个人设置</h2>
    </div>
    <div class="card-body">
        <?php if ($error): ?>
            <div class="alert alert-error"><?php echo e($error); ?></div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="alert alert-success"><?php echo e($success); ?></div>
        <?php endif; ?>

        <!-- 头像设置 - 单独栏目 -->
        <div style="margin-bottom: 40px;">
            <h3 style="margin-bottom: 20px; padding-bottom: 10px; border-bottom: 2px solid #2563eb; display: inline-block;">
                <i class="fas fa-camera"></i> 头像设置
            </h3>
            
            <div style="display: grid; grid-template-columns: 200px 1fr; gap: 30px; align-items: start;">
                <!-- 当前头像预览 -->
                <div style="text-align: center;">
                    <p style="color: #666; font-size: 14px; margin-bottom: 10px;">当前头像</p>
                    <div style="width: 150px; height: 150px; border-radius: 50%; margin: 0 auto; background: #f0f0f0; display: flex; align-items: center; justify-content: center; overflow: hidden; border: 3px solid #e0e0e0;">
                        <?php if ($user['avatar']): ?>
                            <img src="<?php echo e($user['avatar']); ?>?t=<?php echo time(); ?>" alt="头像" id="current-avatar" style="width: 100%; height: 100%; object-fit: cover;">
                        <?php else: ?>
                            <i class="fas fa-user" id="current-avatar-icon" style="font-size: 60px; color: #999;"></i>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- 头像裁剪区域 -->
                <div>
                    <div style="margin-bottom: 20px;">
                        <input type="file" id="avatar-input" class="form-control" accept="image/jpeg,image/png,image/gif,image/webp" style="display: none;">
                        <button type="button" class="btn btn-primary" onclick="document.getElementById('avatar-input').click()">
                            <i class="fas fa-folder-open"></i> 选择图片
                        </button>
                        <span style="color: #666; font-size: 14px; margin-left: 10px;">支持 JPG、PNG、GIF、WEBP 格式，最大 5MB</span>
                    </div>
                    
                    <!-- 裁剪容器 -->
                    <div id="cropper-container" style="display: none; margin-bottom: 20px;">
                        <div style="max-width: 400px; max-height: 400px; margin-bottom: 15px;">
                            <img id="cropper-image" src="" alt="待裁剪图片" style="max-width: 100%; display: block;">
                        </div>
                        <div style="display: flex; gap: 10px;">
                            <button type="button" class="btn btn-primary" id="crop-btn">
                                <i class="fas fa-crop"></i> 确认裁剪并上传
                            </button>
                            <button type="button" class="btn btn-outline" onclick="resetCropper()">
                                <i class="fas fa-times"></i> 取消
                            </button>
                        </div>
                    </div>
                    
                    <!-- 裁剪预览 -->
                    <div id="preview-container" style="display: none;">
                        <p style="color: #666; font-size: 14px; margin-bottom: 10px;">预览效果：</p>
                        <div style="width: 150px; height: 150px; border-radius: 50%; overflow: hidden; border: 3px solid #2563eb;">
                            <img id="preview-image" src="" alt="预览" style="width: 100%; height: 100%; object-fit: cover;">
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- 隐藏表单用于提交裁剪后的图片 -->
            <form method="POST" action="" id="avatar-form" style="display: none;">
                <input type="hidden" name="action" value="avatar">
                <input type="hidden" name="csrf_token" value="<?php echo generateCsrfToken(); ?>">
                <input type="hidden" name="avatar_data" id="avatar-data">
            </form>
        </div>
        
        <hr style="margin: 30px 0; border: none; border-top: 1px solid #e0e0e0;">

        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px;">
            <!-- 个人资料 -->
            <div>
                <h3 style="margin-bottom: 20px;"><i class="fas fa-user-edit"></i> 修改个人资料</h3>
                <form method="POST" action="">
                    <input type="hidden" name="action" value="profile">
                    <input type="hidden" name="csrf_token" value="<?php echo generateCsrfToken(); ?>">
                    
                    <div class="form-group">
                        <label>用户名</label>
                        <input type="text" class="form-control" value="<?php echo e($user['username']); ?>" disabled>
                        <small class="form-text">用户名不能修改</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="email">邮箱</label>
                        <input type="email" id="email" name="email" class="form-control" required
                               value="<?php echo e($user['email']); ?>">
                    </div>
                    
                    <div class="form-group">
                        <label for="signature">个性签名</label>
                        <textarea id="signature" name="signature" class="form-control" rows="3"
                                  placeholder="写点什么来展示你自己..."><?php echo e($user['signature']); ?></textarea>
                        <small class="form-text">最多500个字符</small>
                    </div>
                    
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i> 保存资料
                    </button>
                </form>
            </div>
            
            <!-- 修改密码 -->
            <div>
                <h3 style="margin-bottom: 20px;"><i class="fas fa-key"></i> 修改密码</h3>
                <form method="POST" action="">
                    <input type="hidden" name="action" value="password">
                    <input type="hidden" name="csrf_token" value="<?php echo generateCsrfToken(); ?>">
                    
                    <div class="form-group">
                        <label for="old_password">当前密码</label>
                        <input type="password" id="old_password" name="old_password" class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="new_password">新密码</label>
                        <input type="password" id="new_password" name="new_password" class="form-control" required>
                        <small class="form-text">至少6位</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="confirm_password">确认新密码</label>
                        <input type="password" id="confirm_password" name="confirm_password" class="form-control" required>
                    </div>
                    
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-key"></i> 修改密码
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- 引入 Cropper.js -->
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/cropperjs/1.5.13/cropper.min.css">
<script src="https://cdnjs.cloudflare.com/ajax/libs/cropperjs/1.5.13/cropper.min.js"></script>

<script>
let cropper = null;
const MAX_FILE_SIZE = 5 * 1024 * 1024; // 5MB

// 文件选择处理
document.getElementById('avatar-input').addEventListener('change', function(e) {
    const file = e.target.files[0];
    
    if (!file) return;
    
    // 检查文件大小
    if (file.size > MAX_FILE_SIZE) {
        alert('上传失败，图片大小超出上限（最大值：5MB）');
        this.value = '';
        return;
    }
    
    // 检查文件类型
    const allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    if (!allowedTypes.includes(file.type)) {
        alert('只支持 JPG、PNG、GIF、WEBP 格式的图片');
        this.value = '';
        return;
    }
    
    // 读取文件并显示裁剪界面
    const reader = new FileReader();
    reader.onload = function(e) {
        const image = document.getElementById('cropper-image');
        image.src = e.target.result;
        
        document.getElementById('cropper-container').style.display = 'block';
        document.getElementById('preview-container').style.display = 'none';
        
        // 初始化 Cropper
        if (cropper) {
            cropper.destroy();
        }
        
        cropper = new Cropper(image, {
            aspectRatio: 1,
            viewMode: 1,
            dragMode: 'move',
            autoCropArea: 0.8,
            restore: false,
            guides: true,
            center: true,
            highlight: false,
            cropBoxMovable: true,
            cropBoxResizable: true,
            toggleDragModeOnDblclick: false,
            ready: function() {
                // 裁剪框准备好后更新预览
                updatePreview();
            },
            crop: function() {
                updatePreview();
            }
        });
    };
    reader.readAsDataURL(file);
});

// 更新预览
function updatePreview() {
    if (!cropper) return;
    
    const canvas = cropper.getCroppedCanvas({
        width: 300,
        height: 300,
        fillColor: '#fff',
        imageSmoothingEnabled: true,
        imageSmoothingQuality: 'high',
    });
    
    const previewImage = document.getElementById('preview-image');
    previewImage.src = canvas.toDataURL('image/png');
    document.getElementById('preview-container').style.display = 'block';
}

// 确认裁剪并上传
document.getElementById('crop-btn').addEventListener('click', function() {
    if (!cropper) return;
    
    const canvas = cropper.getCroppedCanvas({
        width: 300,
        height: 300,
        fillColor: '#fff',
        imageSmoothingEnabled: true,
        imageSmoothingQuality: 'high',
    });
    
    // 转换为 base64
    const imageData = canvas.toDataURL('image/png');
    
    // 检查裁剪后的大小（base64 大约比原图大 33%）
    const base64Length = imageData.length - 'data:image/png;base64,'.length;
    const sizeInBytes = (base64Length * 3) / 4;
    
    if (sizeInBytes > MAX_FILE_SIZE) {
        alert('裁剪后的图片仍然超出上限（最大值：5MB），请尝试压缩原图或选择更小的区域');
        return;
    }
    
    // 填充表单并提交
    document.getElementById('avatar-data').value = imageData;
    document.getElementById('avatar-form').submit();
});

// 重置裁剪器
function resetCropper() {
    if (cropper) {
        cropper.destroy();
        cropper = null;
    }
    document.getElementById('avatar-input').value = '';
    document.getElementById('cropper-container').style.display = 'none';
    document.getElementById('preview-container').style.display = 'none';
}
</script>

<style>
/* Cropper 样式调整 */
.cropper-container {
    direction: ltr;
    font-size: 0;
    line-height: 0;
    position: relative;
    touch-action: none;
    user-select: none;
}

.cropper-modal {
    background-color: #000;
    opacity: 0.5;
}

.cropper-crop-box {
    cursor: move;
}

.cropper-view-box {
    display: block;
    height: 100%;
    outline: 1px solid #2563eb;
    outline-color: rgba(37, 99, 235, 0.75);
    overflow: hidden;
    width: 100%;
}

.cropper-dashed {
    border: 0 dashed #eee;
    display: block;
    opacity: 0.5;
    position: absolute;
}

.cropper-dashed.dashed-h {
    border-bottom-width: 1px;
    border-top-width: 1px;
    height: calc(100% / 3);
    left: 0;
    top: calc(100% / 3);
    width: 100%;
}

.cropper-dashed.dashed-v {
    border-left-width: 1px;
    border-right-width: 1px;
    height: 100%;
    left: calc(100% / 3);
    top: 0;
    width: calc(100% / 3);
}

.cropper-center {
    display: block;
    height: 0;
    left: 50%;
    opacity: 0.75;
    position: absolute;
    top: 50%;
    width: 0;
}

.cropper-center::before,
.cropper-center::after {
    background-color: #eee;
    content: ' ';
    display: block;
    position: absolute;
}

.cropper-center::before {
    height: 1px;
    left: -3px;
    top: 0;
    width: 7px;
}

.cropper-center::after {
    height: 7px;
    left: 0;
    top: -3px;
    width: 1px;
}

.cropper-face,
.cropper-line,
.cropper-point {
    display: block;
    height: 100%;
    opacity: 0.1;
    position: absolute;
    width: 100%;
}

.cropper-face {
    background-color: #fff;
    left: 0;
    top: 0;
}

.cropper-line {
    background-color: #2563eb;
}

.cropper-line.line-e {
    cursor: ew-resize;
    right: -3px;
    top: 0;
    width: 5px;
}

.cropper-line.line-n {
    cursor: ns-resize;
    height: 5px;
    left: 0;
    top: -3px;
}

.cropper-line.line-w {
    cursor: ew-resize;
    left: -3px;
    top: 0;
    width: 5px;
}

.cropper-line.line-s {
    bottom: -3px;
    cursor: ns-resize;
    height: 5px;
    left: 0;
}

.cropper-point {
    background-color: #2563eb;
    height: 5px;
    opacity: 0.75;
    width: 5px;
}

.cropper-point.point-e {
    cursor: ew-resize;
    margin-top: -3px;
    right: -3px;
    top: 50%;
}

.cropper-point.point-n {
    cursor: ns-resize;
    left: 50%;
    margin-left: -3px;
    top: -3px;
}

.cropper-point.point-w {
    cursor: ew-resize;
    left: -3px;
    margin-top: -3px;
    top: 50%;
}

.cropper-point.point-s {
    bottom: -3px;
    cursor: s-resize;
    left: 50%;
    margin-left: -3px;
}

.cropper-point.point-ne {
    cursor: nesw-resize;
    right: -3px;
    top: -3px;
}

.cropper-point.point-nw {
    cursor: nwse-resize;
    left: -3px;
    top: -3px;
}

.cropper-point.point-sw {
    bottom: -3px;
    cursor: nesw-resize;
    left: -3px;
}

.cropper-point.point-se {
    bottom: -3px;
    cursor: nwse-resize;
    height: 20px;
    opacity: 1;
    right: -3px;
    width: 20px;
}

.cropper-bg {
    background-image: url('data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAQAQMAAAAlPW0iAAAAA3NCSVQICAjb4U/gAAAABlBMVEXMzMz////TjRV2AAAACXBIWXMAAArrAAAK6wGCiw1aAAAAHHRFWHRTb2Z0d2FyZQBBZG9iZSBGaXJld29ya3MgQ1M26LyyjAAAABFJREFUCJlj+M/AgBVhF/0PAH6/D/HkDxOGAAAAAElFTkSuQmCC');
}
</style>

<?php include __DIR__ . '/templates/footer.php'; ?>
